﻿using Org.BouncyCastle.Crypto.Parameters;
using Org.BouncyCastle.Security;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Security.Cryptography;
using System.Text;
using System.Threading.Tasks;

namespace Demo.ConsoleApp.RSADemo
{
    public class RsaHelper
    {
        readonly RSA _privateKeyRsaProvider;
        readonly RSA _publicKeyRsaProvider;
        readonly HashAlgorithmName _hashAlgorithmName;
        readonly Encoding _encoding;

        public RsaHelper(string publicKey, string privateKey, bool isJavaKey = false) 
            : this(Encoding.UTF8, HashAlgorithmName.SHA256, publicKey, privateKey, isJavaKey)
        {

        }

        public RsaHelper(
            Encoding encoding,
            HashAlgorithmName hashAlgorithmName,
            string publicKey = null, 
            string privateKey = null,
            bool isJavaKey = false
        )
        {
            _encoding = encoding;
            _hashAlgorithmName = hashAlgorithmName;

            if (!string.IsNullOrEmpty(publicKey))
            {
                if (isJavaKey)
                    _publicKeyRsaProvider = CreateRsaProviderFromPublicKeyJava(publicKey);
                else
                    _publicKeyRsaProvider = CreateRsaProviderFromPublicKey(publicKey);
            }

            if (!string.IsNullOrEmpty(privateKey))
            {
                if (isJavaKey)
                    _privateKeyRsaProvider = CreateRsaProviderFromPrivateKeyJava(privateKey);
                else
                    _privateKeyRsaProvider = CreateRsaProviderFromPrivateKey(privateKey);
            }
            
        }

        public string Sign(string input)
        {
            var buffer = _encoding.GetBytes(input);
            var signBuffer = Sign(buffer);
            return Convert.ToBase64String(signBuffer);
        }

        public byte[] Sign(byte[] buffer)
        {
            if (_privateKeyRsaProvider == null)
            {
                throw new Exception("_privateKeyRsaProvider is null");
            }

            return _privateKeyRsaProvider.SignData(buffer, _hashAlgorithmName, RSASignaturePadding.Pkcs1);
        }

        public bool Verify(string input, string sign)
        {
            var signBuffer = Convert.FromBase64String(sign);
            return Verify(input, signBuffer);
        }

        public bool Verify(string input, byte[] signBuffer)
        {
            if (_publicKeyRsaProvider == null)
            {
                throw new Exception("_publicKeyRsaProvider is null");
            }

            var dataBuffer = _encoding.GetBytes(input);
            var verify = _publicKeyRsaProvider.VerifyData(dataBuffer, signBuffer, _hashAlgorithmName, RSASignaturePadding.Pkcs1);
            return verify;
        }

        public string Encrypt(string input)
        {
            var inputBuffer = Encoding.UTF8.GetBytes(input);

            var base64Data = Convert.ToBase64String(EncryptBytes(inputBuffer));
            return base64Data;
        }

        public byte[] EncryptBytes(byte[] inputBuffer)
        {
            if (_publicKeyRsaProvider == null)
            {
                throw new Exception("_publicKeyRsaProvider is null");
            }

            var buffer = _publicKeyRsaProvider.Encrypt(inputBuffer, RSAEncryptionPadding.Pkcs1);
            return buffer;
        }

        public string Decrypt(string encryptInput)
        {
            var encryptBuffer = Convert.FromBase64String(encryptInput);

            var data = Encoding.UTF8.GetString(DecryptBytes(encryptBuffer));
            return data;
        }

        public byte[] DecryptBytes(byte[] encryptBuffer)
        {
            if (_privateKeyRsaProvider == null)
            {
                throw new Exception("_privateKeyRsaProvider is null");
            }

            var buffer = _privateKeyRsaProvider.Decrypt(encryptBuffer, RSAEncryptionPadding.Pkcs1);
            return buffer;
        }

        private RSA CreateRsaProviderFromPublicKeyJava(string publicKey)
        {
            var rsa = RSA.Create();
            var publicKeyXml = GetPublicKeyXml(publicKey);
            rsa.FromXmlString(publicKeyXml);
            return rsa;
        }

        private RSA CreateRsaProviderFromPrivateKeyJava(string privateKey)
        {
            var rsa = RSA.Create();
            var primaryKeyXml = GetPrivateKeyXml(privateKey);
            rsa.FromXmlString(primaryKeyXml);
            return rsa;
        }

        #region 使用私钥创建RSA实例

        private RSA CreateRsaProviderFromPrivateKey(string privateKey)
        {
            var privateKeyBits = System.Convert.FromBase64String(privateKey);

            var rsa = RSA.Create();
            var rsaParameters = new RSAParameters();

            using (BinaryReader binr = new BinaryReader(new MemoryStream(privateKeyBits)))
            {
                byte bt = 0;
                ushort twobytes = 0;
                twobytes = binr.ReadUInt16();
                if (twobytes == 0x8130)
                    binr.ReadByte();
                else if (twobytes == 0x8230)
                    binr.ReadInt16();
                else
                    throw new Exception("Unexpected value read binr.ReadUInt16()");

                twobytes = binr.ReadUInt16();
                if (twobytes != 0x0102)
                    throw new Exception("Unexpected version");

                bt = binr.ReadByte();
                if (bt != 0x00)
                    throw new Exception("Unexpected value read binr.ReadByte()");

                rsaParameters.Modulus = binr.ReadBytes(GetIntegerSize(binr));
                rsaParameters.Exponent = binr.ReadBytes(GetIntegerSize(binr));
                rsaParameters.D = binr.ReadBytes(GetIntegerSize(binr));
                rsaParameters.P = binr.ReadBytes(GetIntegerSize(binr));
                rsaParameters.Q = binr.ReadBytes(GetIntegerSize(binr));
                rsaParameters.DP = binr.ReadBytes(GetIntegerSize(binr));
                rsaParameters.DQ = binr.ReadBytes(GetIntegerSize(binr));
                rsaParameters.InverseQ = binr.ReadBytes(GetIntegerSize(binr));
            }

            rsa.ImportParameters(rsaParameters);
            return rsa;
        }

        #endregion

        #region 使用公钥创建RSA实例

        private RSA CreateRsaProviderFromPublicKey(string publicKeyString)
        {
            // encoded OID sequence for  PKCS #1 rsaEncryption szOID_RSA_RSA = "1.2.840.113549.1.1.1"
            byte[] seqOid = { 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x01, 0x05, 0x00 };
            byte[] seq = new byte[15];

            var x509Key = System.Convert.FromBase64String(publicKeyString);

            // ---------  Set up stream to read the asn.1 encoded SubjectPublicKeyInfo blob  ------
            using (MemoryStream mem = new MemoryStream(x509Key))
            {
                using (BinaryReader binr = new BinaryReader(mem))  //wrap Memory Stream with BinaryReader for easy reading
                {
                    byte bt = 0;
                    ushort twobytes = 0;

                    twobytes = binr.ReadUInt16();
                    if (twobytes == 0x8130) //data read as little endian order (actual data order for Sequence is 30 81)
                        binr.ReadByte();    //advance 1 byte
                    else if (twobytes == 0x8230)
                        binr.ReadInt16();   //advance 2 bytes
                    else
                        return null;

                    seq = binr.ReadBytes(15);       //read the Sequence OID
                    if (!CompareBytearrays(seq, seqOid))    //make sure Sequence for OID is correct
                        return null;

                    twobytes = binr.ReadUInt16();
                    if (twobytes == 0x8103) //data read as little endian order (actual data order for Bit String is 03 81)
                        binr.ReadByte();    //advance 1 byte
                    else if (twobytes == 0x8203)
                        binr.ReadInt16();   //advance 2 bytes
                    else
                        return null;

                    bt = binr.ReadByte();
                    if (bt != 0x00)     //expect null byte next
                        return null;

                    twobytes = binr.ReadUInt16();
                    if (twobytes == 0x8130) //data read as little endian order (actual data order for Sequence is 30 81)
                        binr.ReadByte();    //advance 1 byte
                    else if (twobytes == 0x8230)
                        binr.ReadInt16();   //advance 2 bytes
                    else
                        return null;

                    twobytes = binr.ReadUInt16();
                    byte lowbyte = 0x00;
                    byte highbyte = 0x00;

                    if (twobytes == 0x8102) //data read as little endian order (actual data order for Integer is 02 81)
                        lowbyte = binr.ReadByte();  // read next bytes which is bytes in modulus
                    else if (twobytes == 0x8202)
                    {
                        highbyte = binr.ReadByte(); //advance 2 bytes
                        lowbyte = binr.ReadByte();
                    }
                    else
                        return null;
                    byte[] modint = { lowbyte, highbyte, 0x00, 0x00 };   //reverse byte order since asn.1 key uses big endian order
                    int modsize = BitConverter.ToInt32(modint, 0);

                    int firstbyte = binr.PeekChar();
                    if (firstbyte == 0x00)
                    {   //if first byte (highest order) of modulus is zero, don't include it
                        binr.ReadByte();    //skip this null byte
                        modsize -= 1;   //reduce modulus buffer size by 1
                    }

                    byte[] modulus = binr.ReadBytes(modsize);   //read the modulus bytes

                    if (binr.ReadByte() != 0x02)            //expect an Integer for the exponent data
                        return null;
                    int expbytes = (int)binr.ReadByte();        // should only need one byte for actual exponent data (for all useful values)
                    byte[] exponent = binr.ReadBytes(expbytes);

                    // ------- create RSACryptoServiceProvider instance and initialize with public key -----
                    var rsa = RSA.Create();
                    RSAParameters rsaKeyInfo = new RSAParameters
                    {
                        Modulus = modulus,
                        Exponent = exponent
                    };
                    rsa.ImportParameters(rsaKeyInfo);

                    return rsa;
                }

            }
        }

        #endregion

        #region 导入密钥算法

        private int GetIntegerSize(BinaryReader binr)
        {
            byte bt = 0;
            int count = 0;
            bt = binr.ReadByte();
            if (bt != 0x02)
                return 0;
            bt = binr.ReadByte();

            if (bt == 0x81)
                count = binr.ReadByte();
            else
            if (bt == 0x82)
            {
                var highbyte = binr.ReadByte();
                var lowbyte = binr.ReadByte();
                byte[] modint = { lowbyte, highbyte, 0x00, 0x00 };
                count = BitConverter.ToInt32(modint, 0);
            }
            else
            {
                count = bt;
            }

            while (binr.ReadByte() == 0x00)
            {
                count -= 1;
            }
            binr.BaseStream.Seek(-1, SeekOrigin.Current);
            return count;
        }

        private bool CompareBytearrays(byte[] a, byte[] b)
        {
            if (a.Length != b.Length)
                return false;
            int i = 0;
            foreach (byte c in a)
            {
                if (c != b[i])
                    return false;
                i++;
            }
            return true;
        }

        #endregion

        private string GetPrivateKeyXml(string privateKey)
        {
            RsaPrivateCrtKeyParameters privateKeyParam = (RsaPrivateCrtKeyParameters)PrivateKeyFactory.CreateKey(Convert.FromBase64String(privateKey));
            string xmlPrivateKey = string.Format("<RSAKeyValue><Modulus>{0}</Modulus><Exponent>{1}</Exponent><P>{2}</P><Q>{3}</Q><DP>{4}</DP><DQ>{5}</DQ><InverseQ>{6}</InverseQ><D>{7}</D></RSAKeyValue>",
                         Convert.ToBase64String(privateKeyParam.Modulus.ToByteArrayUnsigned()),
                         Convert.ToBase64String(privateKeyParam.PublicExponent.ToByteArrayUnsigned()),
                         Convert.ToBase64String(privateKeyParam.P.ToByteArrayUnsigned()),
                         Convert.ToBase64String(privateKeyParam.Q.ToByteArrayUnsigned()),
                         Convert.ToBase64String(privateKeyParam.DP.ToByteArrayUnsigned()),
                         Convert.ToBase64String(privateKeyParam.DQ.ToByteArrayUnsigned()),
                         Convert.ToBase64String(privateKeyParam.QInv.ToByteArrayUnsigned()),
                         Convert.ToBase64String(privateKeyParam.Exponent.ToByteArrayUnsigned()));
            return xmlPrivateKey;
        }

        private string GetPublicKeyXml(string publicKey)
        {
            RsaKeyParameters publicKeyParam = (RsaKeyParameters)PublicKeyFactory.CreateKey(Convert.FromBase64String(publicKey));
            string xmlPublicKey = string.Format("<RSAKeyValue><Modulus>{0}</Modulus><Exponent>{1}</Exponent></RSAKeyValue>",
                         Convert.ToBase64String(publicKeyParam.Modulus.ToByteArrayUnsigned()),
                         Convert.ToBase64String(publicKeyParam.Exponent.ToByteArrayUnsigned()));
            return xmlPublicKey;
        }
    }
}
